﻿using System;
using System.Collections.Generic;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Linq;
using System.Numerics;
using System.Reflection;
using System.Runtime.CompilerServices;
using System.Text;
using System.Threading.Tasks;

namespace graphicbox2d
{

    /// <summary>
    /// Vector2 型の拡張メソッドを提供します。
    /// </summary>
    public static class Vector2Extensions
    {
        /// <summary>
        /// Vector2 を PointF に変換します。
        /// </summary>
        /// <param name="v">変換対象の Vector2。</param>
        /// <returns>変換後の PointF。</returns>
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public static PointF ToPointF(this Vector2 v)
        {
            return new PointF(v.X, v.Y);
        }

        /// <summary>
        /// Vector2 を Point に変換します。
        /// </summary>
        /// <param name="v">変換対象の Vector2。</param>
        /// <returns>変換後の Point。</returns>
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public static Point ToPoint(this Vector2 v)
        {
            return new Point((int)v.X, (int)v.Y);
        }
    }

    /// <summary>
    /// PointF 型の拡張メソッドを提供します。
    /// </summary>
    public static class PointFExtensions
    {
        /// <summary>
        /// PointF を Vector2 に変換します。
        /// </summary>
        /// <param name="p">変換対象の PointF。</param>
        /// <returns>変換後の Vector2。</returns>
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public static Vector2 ToVector2(this PointF p)
        {
            return new Vector2(p.X, p.Y);
        }
    }

    /// <summary>
    /// Point 型の拡張メソッドを提供します。
    /// </summary>
    public static class PointExtensions
    {
        /// <summary>
        /// Point を Vector2 に変換します。
        /// </summary>
        /// <param name="p">変換対象の Point。</param>
        /// <returns>変換後の Vector2。</returns>
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public static Vector2 ToVector2(this Point p)
        {
            return new Vector2(p.X, p.Y);
        }
    }

    /// <summary>
    /// Graphics クラス用の拡張メソッド
    /// </summary>
    internal static class GraphicsExtensions
    {
        /// <summary>
        /// 輪郭線描画用ペン
        /// </summary>
        private static readonly Pen m_FillOutLinePen = new Pen(Color.White);

        /// <summary>
        /// コンストラクタ
        /// </summary>
        static GraphicsExtensions()
        {
            m_FillOutLinePen.Width = 1;
            m_FillOutLinePen.DashStyle = DashStyle.Solid;
        }

        /// <summary>
        /// 中心座標と半径を指定して円の輪郭を描画する。
        /// </summary>
        /// <param name="g">描画先の Graphics オブジェクト。</param>
        /// <param name="pen">円の輪郭に使用する Pen。</param>
        /// <param name="cx">円の中心の X 座標。</param>
        /// <param name="cy">円の中心の Y 座標。</param>
        /// <param name="radius">円の半径。</param>
        public static void DrawCircle(this Graphics g, Pen pen, float cx, float cy, float radius)
        {
            g.DrawEllipse(pen, cx - radius, cy - radius, radius * 2, radius * 2);
        }

        /// <summary>
        /// 中心座標と半径を指定して円を塗りつぶし描画する。
        /// </summary>
        /// <param name="g">描画先の Graphics オブジェクト。</param>
        /// <param name="brush">塗りつぶしに使用する Brush。</param>
        /// <param name="cx">円の中心の X 座標。</param>
        /// <param name="cy">円の中心の Y 座標。</param>
        /// <param name="radius">円の半径。</param>
        public static void DrawFillCircle(this Graphics g, Brush brush, float cx, float cy, float radius)
        {
            g.FillEllipse(brush, cx - radius, cy - radius, radius * 2, radius * 2);
        }

        /// <summary>
        /// 塗りつぶし円を描画し、Brush が HatchBrush の場合は前景色で輪郭線を描画する。
        /// </summary>
        /// <param name="g">描画先の Graphics オブジェクト。</param>
        /// <param name="brush">塗りつぶしに使用する Brush。HatchBrush の場合に輪郭が描画される。</param>
        /// <param name="cx">円の中心の X 座標。</param>
        /// <param name="cy">円の中心の Y 座標。</param>
        /// <param name="radius">円の半径。</param>
        public static void DrawFillCircleWithOutLine(this Graphics g, Brush brush, float cx, float cy, float radius)
        {
            g.FillEllipse(brush, cx - radius, cy - radius, radius * 2, radius * 2);

            if (brush is HatchBrush)
            {
                m_FillOutLinePen.Color = (brush as HatchBrush).ForegroundColor;

                DrawCircle(g, m_FillOutLinePen, cx, cy, radius);
            }
        }

        /// <summary>
        /// 多角形を塗りつぶし、Brush が HatchBrush の場合は前景色で輪郭線を描画する。
        /// </summary>
        /// <param name="g">描画先の Graphics オブジェクト。</param>
        /// <param name="brush">塗りつぶしに使用する Brush。HatchBrush の場合に輪郭が描画される。</param>
        /// <param name="points">多角形を構成する頂点座標の配列。</param>
        public static void FillPolygonWithOutLine(this Graphics g, Brush brush, PointF[] points)
        {
            g.FillPolygon(brush, points);

            if (brush is HatchBrush)
            {
                m_FillOutLinePen.Color = (brush as HatchBrush).ForegroundColor;

                g.DrawPolygon(m_FillOutLinePen, points);
            }
        }

        /// <summary>
        /// 指定されたテキストデータを描画します。
        /// </summary>
        /// <param name="g">描画対象の Graphics オブジェクト。</param>
        /// <param name="textData">描画するテキストデータ。</param>
        /// <param name="font">テキスト描画に使用するフォント。</param>
        /// <param name="brush">テキスト描画に使用するブラシ。</param>
        /// <remarks>
        /// <see cref="TextData.Text"/> を <see cref="TextData.TextPoint"/> の位置に描画します。
        /// </remarks>
        public static void DrawText(this Graphics g, TextData textData, Font font, Brush brush)
        {
            g.DrawString(textData.Text, font, brush, textData.TextPoint);
        }

        /// <summary>
        /// 複数のテキストデータを順に描画します。
        /// </summary>
        /// <param name="g">描画対象の Graphics オブジェクト。</param>
        /// <param name="textData">描画するテキストデータのリスト。</param>
        /// <param name="font">テキスト描画に使用するフォント。</param>
        /// <param name="brush">テキスト描画に使用するブラシ。</param>
        /// <remarks>
        /// <see cref="DrawText(Graphics, TextData, Font, Brush)"/> を内部で呼び出し、
        /// リスト内の各テキストを順に描画します。
        /// </remarks>
        public static void DrawTexts(this Graphics g, List<TextData> textData, Font font, Brush brush)
        {
            for (int i = 0; i < textData.Count; i++)
            {
                DrawText(g, textData[i], font, brush);
            }
        }

        /// <summary>
        /// 指定されたパラメータで円弧を描画する関数。
        /// </summary>
        /// <param name="g">描画対象の Graphics オブジェクト。</param>
        /// <param name="pen">円弧を描画する Pen。</param>
        /// <param name="center">円の中心座標。</param>
        /// <param name="radius">円の半径。</param>
        /// <param name="startAngle">円弧の開始角度（度数法）。</param>
        /// <param name="endAngle">円弧の終了角度（度数法）。</param>
        /// <param name="IsDrawSideLine">円弧両サイドの線の描画有無</param>
        public static void DrawArc(this Graphics g, Pen pen, PointF center, float radius, float startAngle, float endAngle, bool IsDrawSideLine)
        {
            // 円弧を描画するための外接矩形を作成
            RectangleF rect = new RectangleF(
                center.X - radius,
                center.Y - radius,
                radius * 2,
                radius * 2);

            float sweepAngle = endAngle - startAngle;

            // 円弧を描画
            g.DrawArc(pen, rect, startAngle, sweepAngle);

            if (IsDrawSideLine == true)
            {
                PointF StartAnglePoint = GetArcSideLinePoint(center, radius, startAngle);

                PointF EndAnglePoint = GetArcSideLinePoint(center, radius, endAngle);

                g.DrawLine(pen, center, StartAnglePoint);

                g.DrawLine(pen, center, EndAnglePoint);
            }
        }

        /// <summary>
        /// 円弧を塗りつぶして描画し、必要に応じてアウトラインも描画する拡張メソッド。
        /// Brush が HatchBrush の場合は、その前景色を用いて円弧の輪郭線を描画する。
        /// </summary>
        /// <param name="g">描画対象の Graphics オブジェクト。</param>
        /// <param name="brush">塗りつぶしに使用する Brush。HatchBrush の場合はアウトラインも描画される。</param>
        /// <param name="center">円の中心座標。</param>
        /// <param name="radius">円の半径。</param>
        /// <param name="startAngle">円弧の開始角度（度数法）。</param>
        /// <param name="endAngle">円弧の終了角度（度数法）。</param>
        /// <param name="IsDrawSideLine">円弧両サイドの線の描画有無</param>
        public static void DrawFillArc(this Graphics g, Brush brush, PointF center, float radius, float startAngle, float endAngle, bool IsDrawSideLine)
        {
            // 円弧を描画するための外接矩形を作成
            Rectangle rect = new Rectangle(
                (int)(center.X - radius),
                (int)(center.Y - radius),
                (int)(radius * 2),
                (int)(radius * 2));

            // 円弧（扇形）を塗りつぶし
            float sweepAngle = endAngle - startAngle;

            g.FillPie(brush, rect, startAngle, sweepAngle);

            if (brush is HatchBrush)
            {
                m_FillOutLinePen.Color = (brush as HatchBrush).ForegroundColor;

                // 円弧の外周を描画
                DrawArc(g, m_FillOutLinePen, center, radius, startAngle, endAngle, IsDrawSideLine);
            }
        }

        /// <summary>
        /// 円弧の角度点（円弧の両端の点）を取得する
        /// </summary>
        /// <param name="center">円弧中点</param>
        /// <param name="radius">円弧半径</param>
        /// <param name="angle">角度</param>
        /// <returns>円弧の両端の点</returns>
        private static PointF GetArcSideLinePoint(PointF center, float radius, float angle)
        {
            float rad = GraphicCaluculate.DegreeToRadian(angle);
            PointF sidePoint = new PointF(
                center.X + radius * (float)Math.Cos(rad),
                center.Y + radius * (float)Math.Sin(rad)
                );

            return sidePoint;
        }
    }

    public static class ColorExtensions
    {
        // 色の反転
        public static Color Invert(this Color color)
        {
            return Color.FromArgb(
                color.A,
                255 - color.R,
                255 - color.G,
                255 - color.B
            );
        }

        // 明るさを調整（係数を掛ける）
        public static Color AdjustBrightness(this Color color, float factor)
        {
            int r = (int)(color.R * factor);
            int g = (int)(color.G * factor);
            int b = (int)(color.B * factor);

            // 0〜255にクリップ
            r = Math.Min(255, Math.Max(0, r));
            g = Math.Min(255, Math.Max(0, g));
            b = Math.Min(255, Math.Max(0, b));

            return Color.FromArgb(color.A, r, g, b);
        }
    }
}
