﻿using System;
using System.Collections;
using System.Collections.Generic;
using System.Drawing;
using System.Linq;
using System.Runtime.InteropServices;
using System.Text;
using System.Threading.Tasks;
using System.Xml;

namespace graphicbox2d.その他
{
    /// <summary>
    /// Xmlデータ変換クラス
    /// </summary>
    static internal class XmlDataConvert
    {
        /// <summary>
        /// 配列データ区切り文字
        /// </summary>
        const char LIST_SPLIT_CHAR = ';';

        /// <summary>
        /// 配列データ開始文字
        /// </summary>
        const char LIST_START = '{';

        /// <summary>
        /// 配列データ終了文字
        /// </summary>
        const char LIST_END   = '}';

        /// <summary>
        /// 任意のデータをXML要素文字列に変換する。
        /// - string型はそのまま返す
        /// - IEnumerable型はリストとして展開して返す
        /// - その他は単一データとして処理する
        /// </summary>
        /// <param name="Data">変換対象のデータオブジェクト</param>
        /// <returns>XML要素文字列</returns>
        public static string DataToElementString(object Data)
        {
            // 文字列型の場合、そのまま返す
            if (typeof(String) == Data.GetType() || typeof(string) == Data.GetType())
            {
                return (string)Data;
            }

            // IEnumerable 型の場合、リストデータとして処理する
            if (Data is IEnumerable enumerable)
            {
                return ListDataToElementString((IEnumerable)Data);
            }

            // 単一データとして処理する
            return ItemDataToElementString(Data);
        }

        /// <summary>
        /// XML要素文字列を指定された型のデータに変換する。
        /// - IEnumerable型の場合はリストに復元
        /// - 単一データの場合は型に応じて復元
        /// </summary>
        /// <param name="ElementString">XML要素文字列</param>
        /// <param name="type">復元対象の型情報</param>
        /// <returns>復元されたオブジェクト</returns>
        public static object ElementStringToData(string ElementString, Type type)
        {
            object RetObject;
            Type elementType = GetEnumerableElementType(type);

            // IEnumerable 型の場合、リストデータとして処理する
            if (elementType != null)
            {
                RetObject = ListElementStringToListData(ElementString, type);
            }
            // 単一データとして処理する
            else
            {
                RetObject = ItemElementStringToData(ElementString, type);
            }
            return RetObject;
        }

        /// <summary>
        /// 単一データをXML要素文字列に変換する。
        /// Color, PointF, List<PointF> は専用処理、それ以外は文字列化。
        /// </summary>
        /// <param name="Data">変換対象の単一データ</param>
        /// <returns>XML要素文字列</returns>
        private static string ItemDataToElementString(object Data)
        {
            Type targetType = Data.GetType();

            string RetString;

            switch (targetType)
            {
                // Color型
                case Type type when type == typeof(Color):
                    RetString = DataToElementString((Color)(object)Data);
                    break;
                // PointF型
                case Type type when type == typeof(PointF):
                    RetString = DataToElementString((PointF)(object)Data);
                    break;
                // その他
                default:
                    // 既定の変換処理を実行
                    RetString = Convert.ToString(Data);
                    break;
            }
            return RetString;
        }

        /// <summary>
        /// Color型をXML要素文字列に変換する（ARGB値を文字列化）。
        /// </summary>
        /// <param name="Data">変換対象のColorデータ</param>
        /// <returns>ARGB値の文字列</returns>
        private static string DataToElementString(Color Data)
        {
            return Data.ToArgb().ToString();
        }

        /// <summary>
        /// PointF型をXML要素文字列に変換する（"(X,Y)"形式）。
        /// </summary>
        /// <param name="Data">変換対象のPointFデータ</param>
        /// <returns>"(X,Y)"形式の文字列</returns>
        private static string DataToElementString(PointF Data)
        {
            return $"({Data.X},{Data.Y})";
        }

        /// <summary>
        /// IEnumerable型をXML要素文字列に変換する。
        /// 各要素をセミコロン区切りで連結し、波括弧で囲む。
        /// </summary>
        /// <param name="Data">変換対象の列挙可能データ</param>
        /// <returns>リスト形式のXML要素文字列</returns>
        private static string ListDataToElementString(IEnumerable Data)
        {
            string ListDataString = "";

            foreach (var item in Data)
            {
                ListDataString += ItemDataToElementString(item) + LIST_SPLIT_CHAR;
            }

            // 末尾のセミコロンを削除
            ListDataString = ListDataString.Substring(0, ListDataString.Length - 1);

            return string.Format("{0}{1}{2}", LIST_START, ListDataString, LIST_END);
        }

        /// <summary>
        /// XML要素文字列を単一データに変換する。
        /// Color, PointF は専用処理、それ以外は型変換。
        /// </summary>
        /// <param name="ElementString">XML要素文字列</param>
        /// <param name="type">復元対象の型情報</param>
        /// <returns>復元されたオブジェクト</returns>
        private static object ItemElementStringToData(string ElementString, Type type)
        {
            object RetObject;
            switch (type)
            {
                // Color型
                case Type t when t == typeof(Color):
                    RetObject = ElementStringToColor(ElementString);
                    break;
                // PointF型
                case Type t when t == typeof(PointF):
                    RetObject = ElementStringToPointF(ElementString);
                    break;
                // その他
                default:
                    RetObject = Convert.ChangeType(ElementString, type);
                    break;
            }

            return RetObject;
        }

        /// <summary>
        /// XML要素文字列をColor型に変換する。
        /// </summary>
        /// <param name="Str">ARGB値を表す文字列</param>
        /// <returns>Colorオブジェクト</returns>
        private static object ElementStringToColor(string Str)
        {
            try
            {
                int argb = int.Parse(Str);
                return Color.FromArgb(argb);
            }
            catch (Exception ex)
            {
                throw new FormatException("Invalid Color format", ex);
            }
        }

        /// <summary>
        /// XML要素文字列をPointF型に変換する。
        /// "(X,Y)"形式をパースしてPointFを生成。
        /// </summary>
        /// <param name="Str">"(X,Y)"形式の文字列</param>
        /// <returns>PointFオブジェクト</returns>
        private static object ElementStringToPointF(string Str)
        {
            try
            {
                string trimmed = Str.Trim(new char[] { '(', ')' });
                string[] parts = trimmed.Split(',');

                float x = float.Parse(parts[0]);
                float y = float.Parse(parts[1]);

                return new PointF(x, y);
            }
            catch (Exception ex)
            {
                throw new FormatException("Invalid PointF format", ex);
            }
        }

        /// <summary>
        /// XML要素文字列をリストデータに変換する。
        /// 要素型を判定し、各要素を復元してリストに追加。
        /// </summary>
        /// <param name="ElementString">リスト形式のXML要素文字列</param>
        /// <param name="type">復元対象のリスト型情報</param>
        /// <returns>復元されたリストオブジェクト</returns>
        private static object ListElementStringToListData(string ElementString, Type type)
        {
            try
            {
                // リストオブジェクトを生成
                Type elementType = GetEnumerableElementType(type);
                Type listType    = typeof(List<>).MakeGenericType(elementType);
                var list         = (IList)Activator.CreateInstance(listType);

                string trimmed = ElementString.Trim(new char[] { LIST_START, LIST_END });
                string[] parts = trimmed.Split(LIST_SPLIT_CHAR);

                foreach (var part in parts)
                {
                    var element = ItemElementStringToData(part, elementType);
                    list.Add(element);
                }

                return list;
            }
            catch (Exception ex)
            {
                throw new FormatException("Invalid PointF format", ex);
            }
        }

        /// <summary>
        /// 指定された型が配列なら要素型を返す。
        /// List<T> など IEnumerable<T> の場合はジェネリック引数を返す。
        /// それ以外は null を返す。
        /// </summary>
        /// <param name="type">判定対象の型情報</param>
        /// <returns>要素型、または null</returns>
        private static Type GetEnumerableElementType(Type type)
        {
            if (type == null)
            {
                return null;
            }

            if (type == typeof(string) || type == typeof(String))
            {
                return null;
            }

            // 配列の場合
            if (type.IsArray)
            {
                return type.GetElementType();
            }

            // ジェネリック IEnumerable<T> の場合
            var enumerableInterface = type.GetInterfaces()
                .FirstOrDefault(i => i.IsGenericType &&
                                     i.GetGenericTypeDefinition() == typeof(IEnumerable<>));

            if (enumerableInterface != null)
            {
                return enumerableInterface.GetGenericArguments()[0];
            }

            return null;
        }
    }
}
